import { expect } from 'chai';
import { describe } from 'mocha';
import { ethers } from 'hardhat';
import { Signers } from './types';
import {
  OfferingToken,
  PaymentToken,
  TradingManagementExecutorFacet,
} from '../../types';
import {
  deployDiamond,
  DiamondAddress,
} from '../../scripts/deployTradingManagement';
import {
  deployContractsForTests,
  FameContracts,
  getSigners,
} from './helpers/deploy-helper';
import { initializeContracts } from './helpers/contracts-initialization-helper';

describe('38 - Purchase Burned Offer Test', function () {
  let offeringToken: OfferingToken;
  let paymentToken: PaymentToken;
  let tradingManagementExecutorFacet: TradingManagementExecutorFacet;
  let contracts: FameContracts;
  let signers: Signers;

  // Signers
  let owner, customer, dataProvider;

  const testOffer = {
    assetId: 'burn-test-asset-38',
    oid: 'burn-test-offer-38',
    tokenUri: 'https://example.com/burn-test-token-38',
    price: ethers.utils.parseEther('10'),
    capDuration: 31536000, // 1 year in seconds
    capDownloads: '', // Empty for subscription
    capVolume: '', // Empty for subscription
    cdsTarget: '{"key": "value"}',
    cdsSl: '{"key": "value"}',
  };

  const initialPaymentTokenBalance = ethers.utils.parseUnits('1000', 18);

  before(async function () {
    // Get signers
    signers = await getSigners();
    owner = signers.admin;
    customer = signers.signer1;
    dataProvider = signers.signer2;

    // Deploy the Diamond contract
    await deployDiamond({
      saveInfo: false,
      showInfo: false,
    });

    // Get TradingManagementExecutorFacet instance using Diamond's address
    tradingManagementExecutorFacet = await ethers.getContractAt(
      'TradingManagementExecutorFacet',
      DiamondAddress,
    );

    // Deploy other contracts
    contracts = await deployContractsForTests({ shouldResetNetwork: false });
    await initializeContracts({ contracts, signers });

    offeringToken = contracts.offeringTokenContract;
    paymentToken = contracts.paymentTokenContract;

    // Set up TradingManagement storage
    const tradingManagementStorageFacet = await ethers.getContractAt(
      'TradingManagementStorageFacet',
      DiamondAddress,
    );

    await tradingManagementStorageFacet
      .connect(owner)
      .setOfferingTokenContract(contracts.offeringTokenContract.address);
    await tradingManagementStorageFacet
      .connect(owner)
      .setPaymentTokenContract(contracts.paymentTokenContract.address);
    await tradingManagementStorageFacet
      .connect(owner)
      .setBourseContractAddress(contracts.bourseContract.address);
    await tradingManagementStorageFacet
      .connect(owner)
      .setPaygDataAccessContract(
        contracts.dataAccessPayAsYouGoContract.address,
      );
    await tradingManagementStorageFacet
      .connect(owner)
      .setPayuDataAccessContract(
        contracts.dataAccessPayAsYouUseContract.address,
      );
    await tradingManagementStorageFacet
      .connect(owner)
      .setSubscriptionDataAccessContract(
        contracts.dataAccessSubscriptionContract.address,
      );

    // Mint payment tokens to customer
    await contracts.governanceContract
      .connect(owner)
      .mintCoin(await customer.getAddress(), 'FDE', initialPaymentTokenBalance);

    // Create test offer
    await offeringToken
      .connect(owner)
      .addAsset(
        testOffer.assetId,
        testOffer.oid,
        testOffer.tokenUri,
        await dataProvider.getAddress(),
        testOffer.price,
        testOffer.capDuration,
        testOffer.capDownloads,
        testOffer.capVolume,
        testOffer.cdsTarget,
        testOffer.cdsSl,
        '0x00',
      );

    // Approve both trading management (Diamond) and bourse to spend payment tokens
    await paymentToken
      .connect(customer)
      .approve(DiamondAddress, ethers.constants.MaxUint256);
    await paymentToken
      .connect(customer)
      .approve(contracts.bourseContract.address, ethers.constants.MaxUint256);
  });

  describe('Purchase prevention after burning:', function () {
    it('should allow purchase before burning offering token', async function () {
      const oid = testOffer.oid;

      // Verify offer exists before purchase
      expect(await offeringToken.tokenExists(oid)).to.be.true;

      const purchaseData = { purchasedAssetId: oid };
      const dataAsBytes = ethers.utils.hexlify(
        ethers.utils.toUtf8Bytes(JSON.stringify(purchaseData)),
      );

      // Purchase should succeed
      await expect(
        tradingManagementExecutorFacet
          .connect(customer)
          .purchaseAccessRightSubscription(
            oid,
            dataAsBytes,
            testOffer.capDuration,
          ),
      ).to.not.be.reverted;
    });

    it('should prevent subscription purchase after burning offering token', async function () {
      // Create a new subscription offer for this test
      const burnTestOffer = {
        assetId: 'burn-sub-asset',
        oid: 'burn-sub-offer',
        tokenUri: 'https://example.com/burn-sub-token',
        price: ethers.utils.parseEther('5'),
        capDuration: 31536000,
        capDownloads: '',
        capVolume: '',
        cdsTarget: '{}',
        cdsSl: '{}',
      };

      await offeringToken
        .connect(owner)
        .addAsset(
          burnTestOffer.assetId,
          burnTestOffer.oid,
          burnTestOffer.tokenUri,
          await dataProvider.getAddress(),
          burnTestOffer.price,
          burnTestOffer.capDuration,
          burnTestOffer.capDownloads,
          burnTestOffer.capVolume,
          burnTestOffer.cdsTarget,
          burnTestOffer.cdsSl,
          '0x00',
        );

      // Verify offer exists before burning
      expect(await offeringToken.tokenExists(burnTestOffer.oid)).to.be.true;

      // Burn the offering token
      await offeringToken.removeOffering(burnTestOffer.oid);

      // Verify offer no longer exists
      expect(await offeringToken.tokenExists(burnTestOffer.oid)).to.be.false;

      const purchaseData = { purchasedAssetId: burnTestOffer.oid };
      const dataAsBytes = ethers.utils.hexlify(
        ethers.utils.toUtf8Bytes(JSON.stringify(purchaseData)),
      );

      // Purchase should fail after burning - using the correct error message
      await expect(
        tradingManagementExecutorFacet
          .connect(customer)
          .purchaseAccessRightSubscription(
            burnTestOffer.oid,
            dataAsBytes,
            burnTestOffer.capDuration,
          ),
      ).to.be.revertedWith("This offer doesn't exist!");
    });

    it('should prevent PAYU purchase after burning offering token', async function () {
      // Create PAYU offer
      const payuOffer = {
        assetId: 'burn-payu-asset-38',
        oid: 'burn-payu-offer-38',
        tokenUri: 'https://example.com/payu-token-38',
        price: ethers.utils.parseEther('5'),
        capDuration: 86400,
        capDownloads: '100', // PAYU has downloads
        capVolume: '',
        cdsTarget: '{}',
        cdsSl: '{}',
      };

      await offeringToken
        .connect(owner)
        .addAsset(
          payuOffer.assetId,
          payuOffer.oid,
          payuOffer.tokenUri,
          await dataProvider.getAddress(),
          payuOffer.price,
          payuOffer.capDuration,
          payuOffer.capDownloads,
          payuOffer.capVolume,
          payuOffer.cdsTarget,
          payuOffer.cdsSl,
          '0x00',
        );

      // Verify PAYU offer exists
      expect(await offeringToken.tokenExists(payuOffer.oid)).to.be.true;

      // Burn the PAYU offering
      await offeringToken.removeOffering(payuOffer.oid);

      // Verify offer no longer exists
      expect(await offeringToken.tokenExists(payuOffer.oid)).to.be.false;

      const purchaseData = { purchasedAssetId: payuOffer.oid };
      const dataAsBytes = ethers.utils.hexlify(
        ethers.utils.toUtf8Bytes(JSON.stringify(purchaseData)),
      );

      // PAYU purchase should fail - using the correct error message
      await expect(
        tradingManagementExecutorFacet
          .connect(customer)
          .purchaseAccessRightPAYU(
            payuOffer.oid,
            dataAsBytes,
            payuOffer.capDuration,
          ),
      ).to.be.revertedWith("This offer doesn't exist!");
    });

    it('should prevent PAYG purchase after burning offering token', async function () {
      // Create PAYG offer
      const paygOffer = {
        assetId: 'burn-payg-asset-38',
        oid: 'burn-payg-offer-38',
        tokenUri: 'https://example.com/payg-token-38',
        price: ethers.utils.parseEther('2'),
        capDuration: 3600,
        capDownloads: '',
        capVolume: '1GB', // PAYG has volume
        cdsTarget: '{}',
        cdsSl: '{}',
      };

      await offeringToken
        .connect(owner)
        .addAsset(
          paygOffer.assetId,
          paygOffer.oid,
          paygOffer.tokenUri,
          await dataProvider.getAddress(),
          paygOffer.price,
          paygOffer.capDuration,
          paygOffer.capDownloads,
          paygOffer.capVolume,
          paygOffer.cdsTarget,
          paygOffer.cdsSl,
          '0x00',
        );

      // Verify PAYG offer exists
      expect(await offeringToken.tokenExists(paygOffer.oid)).to.be.true;

      // Burn the PAYG offering
      await offeringToken.removeOffering(paygOffer.oid);

      // Verify offer no longer exists
      expect(await offeringToken.tokenExists(paygOffer.oid)).to.be.false;

      const purchaseData = { purchasedAssetId: paygOffer.oid };
      const dataAsBytes = ethers.utils.hexlify(
        ethers.utils.toUtf8Bytes(JSON.stringify(purchaseData)),
      );

      // PAYG purchase should fail - using the correct error message
      await expect(
        tradingManagementExecutorFacet
          .connect(customer)
          .purchaseAccessRightPAYG(
            paygOffer.oid,
            dataAsBytes,
            paygOffer.capDuration,
          ),
      ).to.be.revertedWith("This offer doesn't exist!");
    });
  });

  describe('Historical data preservation after burning:', function () {
    it('should preserve historical data but prevent purchase', async function () {
      // Create a new offer for this test
      const historicalTestOffer = {
        assetId: 'historical-test-asset',
        oid: 'historical-test-offer',
        tokenUri: 'https://example.com/historical-token',
        price: ethers.utils.parseEther('3'),
        capDuration: 86400,
        capDownloads: '',
        capVolume: '',
        cdsTarget: '{}',
        cdsSl: '{}',
      };

      await offeringToken
        .connect(owner)
        .addAsset(
          historicalTestOffer.assetId,
          historicalTestOffer.oid,
          historicalTestOffer.tokenUri,
          await dataProvider.getAddress(),
          historicalTestOffer.price,
          historicalTestOffer.capDuration,
          historicalTestOffer.capDownloads,
          historicalTestOffer.capVolume,
          historicalTestOffer.cdsTarget,
          historicalTestOffer.cdsSl,
          '0x00',
        );

      // Burn the offer
      await offeringToken.removeOffering(historicalTestOffer.oid);

      // Verify offer exists historically but not currently
      expect(
        await offeringToken.historicallyTokenExists(historicalTestOffer.oid),
      ).to.be.true;
      expect(await offeringToken.tokenExists(historicalTestOffer.oid)).to.be
        .false;

      // Verify offering info is still accessible
      const offerInfo = await offeringToken.getOfferIdInfo(
        historicalTestOffer.oid,
      );
      expect(offerInfo.oid).to.equal(historicalTestOffer.oid);
      expect(offerInfo.dataAccessPrice).to.equal(historicalTestOffer.price);

      // But purchase should still fail - using the correct error message
      const purchaseData = { purchasedAssetId: historicalTestOffer.oid };
      const dataAsBytes = ethers.utils.hexlify(
        ethers.utils.toUtf8Bytes(JSON.stringify(purchaseData)),
      );

      await expect(
        tradingManagementExecutorFacet
          .connect(customer)
          .purchaseAccessRightSubscription(
            historicalTestOffer.oid,
            dataAsBytes,
            historicalTestOffer.capDuration,
          ),
      ).to.be.revertedWith("This offer doesn't exist!");
    });

    it('should show burned token in historical records but not in current minted tokens', async function () {
      // Create a new offer for this test
      const mintedTestOffer = {
        assetId: 'minted-test-asset',
        oid: 'minted-test-offer',
        tokenUri: 'https://example.com/minted-token',
        price: ethers.utils.parseEther('1'),
        capDuration: 3600,
        capDownloads: '',
        capVolume: '',
        cdsTarget: '{}',
        cdsSl: '{}',
      };

      await offeringToken
        .connect(owner)
        .addAsset(
          mintedTestOffer.assetId,
          mintedTestOffer.oid,
          mintedTestOffer.tokenUri,
          await dataProvider.getAddress(),
          mintedTestOffer.price,
          mintedTestOffer.capDuration,
          mintedTestOffer.capDownloads,
          mintedTestOffer.capVolume,
          mintedTestOffer.cdsTarget,
          mintedTestOffer.cdsSl,
          '0x00',
        );

      const oidHash = await offeringToken.getIDHash(mintedTestOffer.oid);

      // Verify token is in both current and historical records before burning
      let mintedTokens = await offeringToken.getMintedTokens();
      let historicalTokens = await offeringToken.getHistoricallyMintedTokens();

      // Convert BigNumbers to strings for comparison
      const mintedTokensStr = mintedTokens.map((token) => token.toString());
      const historicalTokensStr = historicalTokens.map((token) =>
        token.toString(),
      );
      const oidHashStr = oidHash.toString();

      expect(mintedTokensStr).to.include(oidHashStr);
      expect(historicalTokensStr).to.include(oidHashStr);

      // Burn the offer
      await offeringToken.removeOffering(mintedTestOffer.oid);

      // Verify token is removed from current minted tokens but remains in historical
      mintedTokens = await offeringToken.getMintedTokens();
      historicalTokens = await offeringToken.getHistoricallyMintedTokens();

      const mintedTokensStrAfter = mintedTokens.map((token) =>
        token.toString(),
      );
      const historicalTokensStrAfter = historicalTokens.map((token) =>
        token.toString(),
      );

      expect(mintedTokensStrAfter).to.not.include(oidHashStr);
      expect(historicalTokensStrAfter).to.include(oidHashStr);
    });
  });

  describe('Purchase validation workflow:', function () {
    it('should handle burned offers gracefully in purchase flow', async function () {
      // Create a new offer for this test
      const validationTestOffer = {
        assetId: 'validation-test-asset',
        oid: 'validation-test-offer',
        tokenUri: 'https://example.com/validation-token',
        price: ethers.utils.parseEther('7'),
        capDuration: 86400,
        capDownloads: '',
        capVolume: '',
        cdsTarget: '{}',
        cdsSl: '{}',
      };

      await offeringToken
        .connect(owner)
        .addAsset(
          validationTestOffer.assetId,
          validationTestOffer.oid,
          validationTestOffer.tokenUri,
          await dataProvider.getAddress(),
          validationTestOffer.price,
          validationTestOffer.capDuration,
          validationTestOffer.capDownloads,
          validationTestOffer.capVolume,
          validationTestOffer.cdsTarget,
          validationTestOffer.cdsSl,
          '0x00',
        );

      // Burn the offer first
      await offeringToken.removeOffering(validationTestOffer.oid);

      // Verify the system properly handles the burned offer
      expect(await offeringToken.tokenExists(validationTestOffer.oid)).to.be
        .false;

      const purchaseData = { purchasedAssetId: validationTestOffer.oid };
      const dataAsBytes = ethers.utils.hexlify(
        ethers.utils.toUtf8Bytes(JSON.stringify(purchaseData)),
      );

      // Purchase attempt should fail gracefully - using the correct error message
      await expect(
        tradingManagementExecutorFacet
          .connect(customer)
          .purchaseAccessRightSubscription(
            validationTestOffer.oid,
            dataAsBytes,
            validationTestOffer.capDuration,
          ),
      ).to.be.revertedWith("This offer doesn't exist!");
    });
  });
});
